// Dashboard-specific functionality and data management

class Dashboard {
    constructor() {
        this.data = {};
        this.charts = {};
        this.refreshInterval = null;
        this.init();
    }

    init() {
        this.loadInitialData();
        this.initializeCharts();
        this.startAutoRefresh();
        this.bindEvents();
    }

    // Load initial dashboard data
    async loadInitialData() {
        try {
            // Simulate API calls for different data sources
            const [violationsData, performanceData, systemData] = await Promise.all([
                this.fetchViolationsData(),
                this.fetchPerformanceData(),
                this.fetchSystemData()
            ]);

            this.data = {
                violations: violationsData,
                performance: performanceData,
                system: systemData
            };

            this.updateDashboard();
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showErrorMessage('Failed to load dashboard data');
        }
    }

    // Simulate violations data fetch
    async fetchViolationsData() {
        // Simulate API delay
        await new Promise(resolve => setTimeout(resolve, 500));
        
        return {
            totalViolations: 2847,
            pendingCases: 145,
            resolutionRate: 95.2,
            totalFines: 487000,
            sectorBreakdown: {
                taxi: { violations: 1245, activeCase: 78, resolutionRate: 94.7 },
                bus: { violations: 892, activeCase: 45, resolutionRate: 96.1 },
                marine: { violations: 710, activeCase: 22, resolutionRate: 95.8 }
            },
            trends: {
                daily: [127, 134, 98, 156, 142, 119, 167],
                weekly: [892, 945, 823, 934, 867, 923, 889]
            }
        };
    }

    // Simulate performance data fetch
    async fetchPerformanceData() {
        await new Promise(resolve => setTimeout(resolve, 300));
        
        return {
            activeDrivers: 12450,
            fleetUtilization: 87.5,
            totalRevenue: 2400000,
            wastedKm: 45230,
            tripStats: {
                completed: 178945,
                averageDuration: 23.5,
                averageDistance: 12.8,
                customerRating: 4.7
            },
            sectorPerformance: {
                taxi: { utilization: 89.3, score: 92, vehicles: 8234 },
                bus: { utilization: 82.7, score: 85, vehicles: 2156 },
                marine: { utilization: 74.2, score: 78, vehicles: 456 }
            }
        };
    }

    // Simulate system data fetch
    async fetchSystemData() {
        await new Promise(resolve => setTimeout(resolve, 200));
        
        return {
            activeVehicles: 45230,
            systemUptime: 99.1,
            violationsToday: 127,
            lastUpdate: new Date(),
            alerts: [
                { type: 'critical', message: 'High-risk driver detected: DR-45789', time: '2 mins ago' },
                { type: 'warning', message: 'Scheduled maintenance at 2:00 AM', time: '1 hour ago' },
                { type: 'info', message: 'Monthly fleet report available', time: '3 hours ago' }
            ]
        };
    }

    // Update dashboard with loaded data
    updateDashboard() {
        this.updateKPICards();
        this.updateOverviewCards();
        this.updateQuickStats();
        this.updateCharts();
        this.updateLastUpdated();
    }

    updateKPICards() {
        // This would update KPI cards on various pages
        const elements = {
            'total-violations': this.data.violations?.totalViolations,
            'pending-cases': this.data.violations?.pendingCases,
            'resolution-rate': this.data.violations?.resolutionRate + '%',
            'total-fines': this.formatCurrency(this.data.violations?.totalFines),
            'active-drivers': this.formatNumber(this.data.performance?.activeDrivers),
            'fleet-utilization': this.data.performance?.fleetUtilization + '%',
            'total-revenue': this.formatCurrency(this.data.performance?.totalRevenue),
            'wasted-km': this.formatNumber(this.data.performance?.wastedKm)
        };

        Object.entries(elements).forEach(([id, value]) => {
            const element = document.getElementById(id);
            if (element && value !== undefined) {
                element.textContent = value;
            }
        });
    }

    updateOverviewCards() {
        // Update overview card statistics
        const violationsCard = document.querySelector('.violations-card .card-stats');
        const performanceCard = document.querySelector('.performance-card .card-stats');

        if (violationsCard && this.data.violations) {
            const activeCases = violationsCard.querySelector('.stat-item:first-child strong');
            const resolutionRate = violationsCard.querySelector('.stat-item:last-child strong');
            
            if (activeCases) activeCases.textContent = this.formatNumber(this.data.violations.pendingCases);
            if (resolutionRate) resolutionRate.textContent = this.data.violations.resolutionRate + '%';
        }

        if (performanceCard && this.data.performance) {
            const activeDrivers = performanceCard.querySelector('.stat-item:first-child strong');
            const fleetUtil = performanceCard.querySelector('.stat-item:last-child strong');
            
            if (activeDrivers) activeDrivers.textContent = this.formatNumber(this.data.performance.activeDrivers);
            if (fleetUtil) fleetUtil.textContent = this.data.performance.fleetUtilization + '%';
        }
    }

    updateQuickStats() {
        const statsElements = {
            'active-vehicles': this.data.system?.activeVehicles,
            'trips-completed': this.data.performance?.tripStats?.completed,
            'system-uptime': this.data.system?.systemUptime + '%',
            'violations-today': this.data.system?.violationsToday
        };

        Object.entries(statsElements).forEach(([className, value]) => {
            const elements = document.querySelectorAll(`.stat-card .stat-number`);
            elements.forEach((element, index) => {
                if (value !== undefined) {
                    switch(index) {
                        case 0:
                            element.textContent = this.formatNumber(this.data.system?.activeVehicles);
                            break;
                        case 1:
                            element.textContent = this.formatNumber(this.data.performance?.tripStats?.completed);
                            break;
                        case 2:
                            element.textContent = this.data.system?.systemUptime + '%';
                            break;
                        case 3:
                            element.textContent = this.data.system?.violationsToday;
                            break;
                    }
                }
            });
        });
    }

    initializeCharts() {
        // Initialize any charts that might be on the dashboard
        this.initializeViolationTrendChart();
        this.initializePerformanceChart();
    }

    initializeViolationTrendChart() {
        const canvas = document.getElementById('violationTrendChart');
        if (!canvas || !this.data.violations) return;

        // Chart initialization would go here
        console.log('Initializing violation trend chart');
    }

    initializePerformanceChart() {
        const canvas = document.getElementById('performanceChart');
        if (!canvas || !this.data.performance) return;

        // Chart initialization would go here
        console.log('Initializing performance chart');
    }

    updateCharts() {
        // Update existing charts with new data
        Object.values(this.charts).forEach(chart => {
            if (chart && chart.update) {
                chart.update();
            }
        });
    }

    updateLastUpdated() {
        const lastUpdatedElements = document.querySelectorAll('.last-update');
        lastUpdatedElements.forEach(element => {
            element.textContent = `Last update: ${this.formatTime(new Date())}`;
        });
    }

    startAutoRefresh() {
        // Refresh data every 5 minutes
        this.refreshInterval = setInterval(() => {
            this.loadInitialData();
        }, 5 * 60 * 1000);
    }

    stopAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    bindEvents() {
        // Handle refresh button clicks
        document.addEventListener('click', (e) => {
            if (e.target.matches('.btn-refresh, .btn-refresh *')) {
                e.preventDefault();
                this.refreshData();
            }
        });
    }

    async refreshData() {
        const refreshBtn = document.querySelector('.btn-refresh');
        if (refreshBtn) {
            refreshBtn.classList.add('loading');
        }

        try {
            await this.loadInitialData();
            this.showSuccessMessage('Data refreshed successfully');
        } catch (error) {
            this.showErrorMessage('Failed to refresh data');
        } finally {
            if (refreshBtn) {
                refreshBtn.classList.remove('loading');
            }
        }
    }

    // Utility methods
    formatNumber(num) {
        if (num >= 1000000) {
            return (num / 1000000).toFixed(1) + 'M';
        } else if (num >= 1000) {
            return (num / 1000).toFixed(1) + 'K';
        }
        return num.toLocaleString();
    }

    formatCurrency(amount) {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    }

    formatTime(date) {
        return new Intl.DateTimeFormat('en-US', {
            hour: '2-digit',
            minute: '2-digit'
        }).format(date);
    }

    showSuccessMessage(message) {
        if (window.showNotification) {
            window.showNotification(message, 'success');
        }
    }

    showErrorMessage(message) {
        if (window.showNotification) {
            window.showNotification(message, 'error');
        }
    }
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    const dashboard = new Dashboard();
    
    // Make dashboard instance available globally for debugging
    window.dashboard = dashboard;
});

// Export dashboard functions
window.refreshDashboard = function() {
    if (window.dashboard) {
        window.dashboard.refreshData();
    }
};