// Main application initialization and global functionality

document.addEventListener('DOMContentLoaded', function() {
    console.log('RTA Dashboard App Initialized');
    
    // Load components
    loadComponent('header', 'components/header.html');
    loadComponent('sidebar', 'components/sidebar.html');
    loadComponent('footer', 'components/footer.html');
    
    // Initialize event listeners
    initializeEventListeners();
    
    // Set active navigation
    setActiveNavigation();
});

// Load HTML components
function loadComponent(elementId, componentPath) {
    const element = document.getElementById(elementId);
    if (element) {
        fetch(componentPath)
            .then(response => response.text())
            .then(html => {
                element.innerHTML = html;
            })
            .catch(error => {
                console.error(`Error loading component ${componentPath}:`, error);
            });
    }
}

// Initialize global event listeners
function initializeEventListeners() {
    // Global search functionality
    document.addEventListener('keyup', function(e) {
        if (e.target.id === 'globalSearch') {
            handleGlobalSearch(e.target.value);
        }
    });
    
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        closeDropdowns(e);
    });
    
    // Handle escape key for modals and dropdowns
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeAllModals();
            closeAllDropdowns();
        }
    });
}

// Set active navigation based on current page
function setActiveNavigation() {
    const currentPath = window.location.pathname;
    const navLinks = document.querySelectorAll('.menu-link');
    
    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href') && currentPath.includes(link.getAttribute('href'))) {
            link.classList.add('active');
        }
    });
}

// Global navigation function
function navigateToSection(section) {
    console.log('Navigating to section:', section);
    
    switch(section) {
        case 'violations':
            window.location.href = 'screens/violations-summary.html';
            break;
        case 'performance':
            window.location.href = 'screens/performance-overview.html';
            break;
        default:
            console.log('Unknown section:', section);
    }
}

// Global search handler
function handleGlobalSearch(query) {
    if (query.length < 3) return;
    
    console.log('Searching for:', query);
    // Implement search functionality here
    // This could search across drivers, vehicles, violations, etc.
}

// Sidebar toggle functionality
function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const content = document.querySelector('.content');
    
    if (sidebar && content) {
        sidebar.classList.toggle('collapsed');
        content.classList.toggle('sidebar-collapsed');
    }
}

// Notification dropdown toggle
function toggleNotifications() {
    const dropdown = document.getElementById('notificationDropdown');
    if (dropdown) {
        dropdown.classList.toggle('show');
    }
}

// User menu dropdown toggle
function toggleUserMenu() {
    const dropdown = document.getElementById('userDropdown');
    if (dropdown) {
        dropdown.classList.toggle('show');
    }
}

// Close dropdowns when clicking outside
function closeDropdowns(event) {
    const notificationDropdown = document.getElementById('notificationDropdown');
    const userDropdown = document.getElementById('userDropdown');
    
    if (notificationDropdown && !event.target.closest('.notifications')) {
        notificationDropdown.classList.remove('show');
    }
    
    if (userDropdown && !event.target.closest('.user-menu')) {
        userDropdown.classList.remove('show');
    }
}

// Close all dropdowns
function closeAllDropdowns() {
    const dropdowns = document.querySelectorAll('.notification-dropdown, .user-dropdown');
    dropdowns.forEach(dropdown => {
        dropdown.classList.remove('show');
    });
}

// Close all modals
function closeAllModals() {
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        modal.classList.remove('show');
    });
}

// Utility function for showing notifications
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span>${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">×</button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

// Utility function for loading states
function setLoadingState(element, loading = true) {
    if (loading) {
        element.classList.add('loading');
        element.setAttribute('disabled', 'disabled');
    } else {
        element.classList.remove('loading');
        element.removeAttribute('disabled');
    }
}

// Format numbers for display
function formatNumber(num) {
    if (num >= 1000000) {
        return (num / 1000000).toFixed(1) + 'M';
    } else if (num >= 1000) {
        return (num / 1000).toFixed(1) + 'K';
    }
    return num.toString();
}

// Format currency
function formatCurrency(amount, currency = 'USD') {
    return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: currency
    }).format(amount);
}

// Format dates
function formatDate(date, format = 'short') {
    const options = {
        short: { month: 'short', day: 'numeric', year: 'numeric' },
        long: { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' },
        time: { hour: '2-digit', minute: '2-digit' }
    };
    
    return new Intl.DateTimeFormat('en-US', options[format]).format(new Date(date));
}

// Export functions for use in other files
window.navigateToSection = navigateToSection;
window.toggleSidebar = toggleSidebar;
window.toggleNotifications = toggleNotifications;
window.toggleUserMenu = toggleUserMenu;
window.showNotification = showNotification;
window.setLoadingState = setLoadingState;
window.formatNumber = formatNumber;
window.formatCurrency = formatCurrency;
window.formatDate = formatDate;