// Navigation and routing functionality for the RTA Dashboard

class Navigation {
    constructor() {
        this.currentPage = '';
        this.navigationHistory = [];
        this.init();
    }

    init() {
        this.bindEvents();
        this.setCurrentPage();
    }

    bindEvents() {
        // Handle navigation link clicks
        document.addEventListener('click', (e) => {
            if (e.target.matches('.menu-link, .menu-link *')) {
                e.preventDefault();
                const link = e.target.closest('.menu-link');
                if (link && link.getAttribute('href')) {
                    this.navigateTo(link.getAttribute('href'));
                }
            }
        });

        // Handle browser back/forward buttons
        window.addEventListener('popstate', (e) => {
            if (e.state && e.state.page) {
                this.loadPage(e.state.page, false);
            }
        });
    }

    navigateTo(url, addToHistory = true) {
        console.log('Navigating to:', url);
        
        // Add to history if not a back/forward navigation
        if (addToHistory) {
            this.navigationHistory.push(this.currentPage);
            history.pushState({ page: url }, '', url);
        }

        this.loadPage(url);
    }

    loadPage(url, updateHistory = true) {
        // Update current page
        this.currentPage = url;
        
        // Update active navigation
        this.updateActiveNavigation(url);
        
        // For SPA-like behavior, you would load content here
        // For now, we just navigate to the URL
        if (updateHistory) {
            window.location.href = url;
        }
    }

    updateActiveNavigation(currentUrl) {
        const menuLinks = document.querySelectorAll('.menu-link');
        
        menuLinks.forEach(link => {
            link.classList.remove('active');
            
            const linkUrl = link.getAttribute('href');
            if (linkUrl && (currentUrl.includes(linkUrl) || linkUrl.includes(currentUrl))) {
                link.classList.add('active');
            }
        });
    }

    setCurrentPage() {
        this.currentPage = window.location.pathname;
        this.updateActiveNavigation(this.currentPage);
    }

    goBack() {
        if (this.navigationHistory.length > 0) {
            const previousPage = this.navigationHistory.pop();
            this.navigateTo(previousPage);
        } else {
            window.history.back();
        }
    }

    // Breadcrumb functionality
    generateBreadcrumb() {
        const pathSegments = window.location.pathname.split('/').filter(segment => segment);
        const breadcrumbContainer = document.querySelector('.breadcrumb');
        
        if (!breadcrumbContainer) return;

        let breadcrumbHTML = '<a href="../index.html"><i class="fas fa-home"></i> Dashboard</a>';
        
        pathSegments.forEach((segment, index) => {
            const isLast = index === pathSegments.length - 1;
            const segmentName = this.formatSegmentName(segment);
            
            if (isLast) {
                breadcrumbHTML += `<span class="separator">></span><span class="current">${segmentName}</span>`;
            } else {
                breadcrumbHTML += `<span class="separator">></span><a href="#">${segmentName}</a>`;
            }
        });
        
        breadcrumbContainer.innerHTML = breadcrumbHTML;
    }

    formatSegmentName(segment) {
        // Convert file names to readable names
        const nameMap = {
            'violations-summary': 'Violations & Enforcement',
            'violation-analysis': 'Violation Analysis',
            'violation-detail-log': 'Violation Detail Log',
            'repeat-offenders': 'Repeat Offenders',
            'performance-overview': 'Driver & Fleet Performance',
            'driver-scorecards': 'Driver Scorecards',
            'fleet-utilization': 'Fleet Utilization',
            'driver-engagement': 'Driver Engagement',
            'comparative-performance': 'Comparative Analysis'
        };

        const baseName = segment.replace('.html', '');
        return nameMap[baseName] || baseName.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }
}

// Quick navigation functions
function navigateToViolations() {
    window.location.href = 'screens/violations-summary.html';
}

function navigateToPerformance() {
    window.location.href = 'screens/performance-overview.html';
}

function navigateToAnalysis() {
    window.location.href = 'screens/violation-analysis.html';
}

function navigateToDetails() {
    window.location.href = 'screens/violation-detail-log.html';
}

function navigateToOffenders() {
    window.location.href = 'screens/repeat-offenders.html';
}

function navigateToDriverScorecard() {
    window.location.href = 'screens/driver-scorecards.html';
}

function navigateToFleetUtilization() {
    window.location.href = 'screens/fleet-utilization.html';
}

function navigateToDriverEngagement() {
    window.location.href = 'screens/driver-engagement.html';
}

function navigateToComparative() {
    window.location.href = 'screens/comparative-performance.html';
}

// Mobile navigation toggle
function toggleMobileNavigation() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('mobile-overlay');
    
    if (sidebar) {
        sidebar.classList.toggle('show');
    }
    
    if (overlay) {
        overlay.classList.toggle('show');
    }
}

// Close mobile navigation
function closeMobileNavigation() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('mobile-overlay');
    
    if (sidebar) {
        sidebar.classList.remove('show');
    }
    
    if (overlay) {
        overlay.classList.remove('show');
    }
}

// Initialize navigation when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    const navigation = new Navigation();
    
    // Generate breadcrumb for current page
    navigation.generateBreadcrumb();
    
    // Handle mobile navigation
    const mobileToggle = document.querySelector('.mobile-nav-toggle');
    if (mobileToggle) {
        mobileToggle.addEventListener('click', toggleMobileNavigation);
    }
    
    // Handle mobile overlay
    const mobileOverlay = document.querySelector('.mobile-overlay');
    if (mobileOverlay) {
        mobileOverlay.addEventListener('click', closeMobileNavigation);
    }
});

// Export navigation functions
window.navigateToViolations = navigateToViolations;
window.navigateToPerformance = navigateToPerformance;
window.navigateToAnalysis = navigateToAnalysis;
window.navigateToDetails = navigateToDetails;
window.navigateToOffenders = navigateToOffenders;
window.navigateToDriverScorecard = navigateToDriverScorecard;
window.navigateToFleetUtilization = navigateToFleetUtilization;
window.navigateToDriverEngagement = navigateToDriverEngagement;
window.navigateToComparative = navigateToComparative;
window.toggleMobileNavigation = toggleMobileNavigation;
window.closeMobileNavigation = closeMobileNavigation;